/**
 * Seletor de Ícones usando Iconify
 * Permite buscar e selecionar ícones de várias bibliotecas
 * Versão: 1.0.1
 */

(function($) {
    'use strict';

    // Bibliotecas de ícones populares
    const ICON_LIBRARIES = [
        { name: 'Material Design Icons', prefix: 'mdi' },
        { name: 'Font Awesome 6', prefix: 'fa6' },
        { name: 'Bootstrap Icons', prefix: 'bi' },
        { name: 'Heroicons', prefix: 'heroicons' },
        { name: 'Feather', prefix: 'feather' },
        { name: 'Tabler Icons', prefix: 'tabler' },
        { name: 'Carbon', prefix: 'carbon' },
        { name: 'Lucide', prefix: 'lucide' }
    ];

    // Cache para resultados de busca
    let searchCache = {};
    let currentTargetField = null;
    let currentPage = 0;
    let isLoading = false;
    let hasMoreIcons = true;
    let currentQuery = '';
    let currentLibrary = 'mdi';

    /**
     * Inicializar seletor de ícones
     */
    function initIconSelector() {
        // Adicionar botão "Escolher Ícone" aos campos de ícone
        $('.upgram-eeat-icon-field, .upgram-years-icon-field, .upgram-orders-icon-field, .upgram-icon-field').each(function() {
            const $field = $(this);
            const $textarea = $field.find('textarea');
            
            // Verificar se já tem botão
            if ($field.find('.upgram-icon-selector-btn').length === 0 && $textarea.length > 0) {
                const $btn = $('<button type="button" class="button upgram-icon-selector-btn" style="margin-top: 5px; margin-left: 5px;">🎨 Escolher Ícone da Biblioteca</button>');
                $textarea.after($btn);
            }
        });

        // Event listener para botões
        $(document).on('click', '.upgram-icon-selector-btn', function(e) {
            e.preventDefault();
            const $btn = $(this);
            const $field = $btn.closest('.upgram-eeat-icon-field, .upgram-years-icon-field, .upgram-orders-icon-field, .upgram-icon-field');
            const $textarea = $field.find('textarea');
            
            if ($textarea.length > 0) {
                currentTargetField = $textarea[0];
                openIconSelector();
            }
        });
    }

    /**
     * Abrir modal de seleção de ícones
     */
    function openIconSelector() {
        // Criar modal se não existir
        if ($('#upgram-icon-selector-modal').length === 0) {
            createIconSelectorModal();
        }

        // Resetar modal
        $('#upgram-icon-selector-search').val('');
        $('#upgram-icon-selector-results').html('<div style="text-align: center; padding: 40px; color: #666;">🔄 Carregando ícones...</div>');
        $('#upgram-icon-selector-library').val('mdi');
        currentPage = 0;
        currentQuery = '';
        currentLibrary = 'mdi';
        hasMoreIcons = true;
        isLoading = false;

        // Mostrar modal
        $('#upgram-icon-selector-modal').fadeIn(200, function() {
            // Após o modal aparecer, carregar ícones
            $('#upgram-icon-selector-search').focus();
            
            // Aguardar um pouco para garantir que o modal está totalmente renderizado
            setTimeout(function() {
                // Aguardar Iconify estar pronto antes de carregar ícones
                if (typeof Iconify !== 'undefined' && typeof Iconify.getIcon === 'function') {
                    // Iconify já está disponível
                    console.log('Iconify já disponível, carregando ícones...');
                    searchIcons('', true);
                } else {
                    // Aguardar Iconify carregar
                    console.log('Aguardando Iconify carregar...');
                    loadIconify().then(function() {
                        console.log('Iconify carregado, carregando ícones...');
                        setTimeout(function() {
                            searchIcons('', true);
                        }, 200);
                    }).catch(function(error) {
                        console.error('Erro ao carregar Iconify:', error);
                        // Mesmo assim, tentar carregar ícones com fallback
                        setTimeout(function() {
                            searchIcons('', true);
                        }, 200);
                    });
                }
            }, 300);
        });
    }

    /**
     * Carregar Iconify
     */
    function loadIconify() {
        return new Promise(function(resolve, reject) {
            // Verificar se Iconify já está disponível
            if (typeof Iconify !== 'undefined' && typeof Iconify.getIcon === 'function') {
                resolve(Iconify);
                return;
            }

            // Verificar se já está sendo carregado
            const existingScript = document.querySelector('script[src*="iconify.design"]');
            if (existingScript) {
                // Aguardar até que o script seja carregado
                let attempts = 0;
                const maxAttempts = 100; // 10 segundos (100 * 100ms)
                
                const checkInterval = setInterval(function() {
                    attempts++;
                    if (typeof Iconify !== 'undefined' && typeof Iconify.getIcon === 'function') {
                        clearInterval(checkInterval);
                        resolve(Iconify);
                    } else if (attempts >= maxAttempts) {
                        clearInterval(checkInterval);
                        reject(new Error('Timeout ao carregar Iconify após 10 segundos'));
                    }
                }, 100);
                
                return;
            }

            // Carregar o script do Iconify
            const script = document.createElement('script');
            script.src = 'https://code.iconify.design/3/3.1.1/iconify.min.js';
            script.async = true;
            
            script.onload = function() {
                // Aguardar um pouco mais para garantir que o Iconify está disponível
                setTimeout(function() {
                    if (typeof Iconify !== 'undefined' && typeof Iconify.getIcon === 'function') {
                        resolve(Iconify);
                    } else {
                        reject(new Error('Iconify foi carregado mas a API não está disponível'));
                    }
                }, 100);
            };
            
            script.onerror = function() {
                reject(new Error('Erro ao carregar o script do Iconify'));
            };
            
            document.head.appendChild(script);
        });
    }

    /**
     * Criar modal de seleção de ícones
     */
    function createIconSelectorModal() {
        const modalHtml = `
            <div id="upgram-icon-selector-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.7); z-index: 100000; overflow-y: auto;">
                <div style="max-width: 900px; margin: 50px auto; background: white; border-radius: 8px; box-shadow: 0 4px 20px rgba(0,0,0,0.3); position: relative;">
                    <div style="padding: 20px; border-bottom: 1px solid #ddd; display: flex; justify-content: space-between; align-items: center;">
                        <h2 style="margin: 0; font-size: 20px; color: #333;">🎨 Seletor de Ícones</h2>
                        <button type="button" id="upgram-icon-selector-close" style="background: none; border: none; font-size: 24px; cursor: pointer; color: #666; padding: 0; width: 30px; height: 30px; line-height: 1;">&times;</button>
                    </div>
                    <div style="padding: 20px;">
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: 600; color: #333;">Biblioteca de Ícones:</label>
                            <select id="upgram-icon-selector-library" style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                                ${ICON_LIBRARIES.map(lib => `<option value="${lib.prefix}">${lib.name}</option>`).join('')}
                            </select>
                        </div>
                        <div style="margin-bottom: 15px;">
                            <label style="display: block; margin-bottom: 5px; font-weight: 600; color: #333;">Buscar Ícone:</label>
                            <input type="text" id="upgram-icon-selector-search" placeholder="Ex: star, heart, check, arrow..." style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 14px;">
                            <small style="display: block; margin-top: 5px; color: #666;">Digite palavras-chave para buscar ícones (ex: star, heart, check)</small>
                        </div>
                        <div id="upgram-icon-selector-results" style="max-height: 500px; overflow-y: auto; border: 1px solid #eee; border-radius: 4px; padding: 15px; background: #f9f9f9; position: relative;">
                            <div style="text-align: center; padding: 40px; color: #666;">🔄 Carregando ícones...</div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        $('body').append(modalHtml);

        // Event listeners
        $('#upgram-icon-selector-close').on('click', closeIconSelector);
        $('#upgram-icon-selector-modal').on('click', function(e) {
            if ($(e.target).attr('id') === 'upgram-icon-selector-modal') {
                closeIconSelector();
            }
        });

        // Busca com debounce
        let searchTimeout;
        $('#upgram-icon-selector-search').on('input', function() {
            clearTimeout(searchTimeout);
            const query = $(this).val().trim();
            searchTimeout = setTimeout(() => {
                if (query.length >= 1) {
                    searchIcons(query, true);
                } else {
                    // Se campo vazio, mostrar ícones populares
                    searchIcons('', true);
                }
            }, 300);
        });

        // Mudança de biblioteca
        $('#upgram-icon-selector-library').on('change', function() {
            const query = $('#upgram-icon-selector-search').val().trim();
            searchIcons(query || '', true);
        });

        // Configurar scroll infinito após criar o modal
        setupInfiniteScroll();

        // Garantir que Iconify está carregado
        loadIconify().catch(function(error) {
            console.error('Erro ao carregar Iconify:', error);
            $('#upgram-icon-selector-results').html('<div style="text-align: center; padding: 40px; color: #d32f2f;">Erro ao carregar biblioteca de ícones. Por favor, recarregue a página.</div>');
        });
    }

    /**
     * Obter lista completa de ícones populares por biblioteca
     */
    function getAllPopularIcons(library) {
        const allIcons = {
            'mdi': ['star', 'star-outline', 'star-half', 'heart', 'heart-outline', 'heart-pulse', 'check', 'check-circle', 'check-all', 'arrow-right', 'arrow-left', 'arrow-up', 'arrow-down', 'arrow-right-bold', 'arrow-left-bold', 'home', 'home-outline', 'home-variant', 'user', 'account', 'account-circle', 'account-outline', 'settings', 'cog', 'cog-outline', 'search', 'magnify', 'menu', 'menu-open', 'close', 'close-circle', 'close-outline', 'edit', 'pencil', 'pencil-outline', 'delete', 'delete-outline', 'delete-forever', 'add', 'plus', 'plus-circle', 'plus-outline', 'remove', 'minus', 'minus-circle', 'download', 'download-outline', 'upload', 'upload-outline', 'lock', 'lock-outline', 'lock-open', 'unlock', 'unlock-outline', 'eye', 'eye-outline', 'eye-off', 'eye-off-outline', 'calendar', 'calendar-outline', 'calendar-month', 'clock', 'clock-outline', 'clock-time-four', 'email', 'email-outline', 'email-open', 'phone', 'phone-outline', 'phone-in-talk', 'map', 'map-outline', 'map-marker', 'shopping', 'shopping-outline', 'cart', 'cart-outline', 'cart-plus', 'credit-card', 'credit-card-outline', 'shield', 'shield-outline', 'shield-check', 'trophy', 'trophy-outline', 'trophy-award', 'award', 'award-outline', 'fire', 'fire-alert', 'lightning-bolt', 'rocket', 'rocket-outline', 'flag', 'flag-outline', 'bookmark', 'bookmark-outline', 'tag', 'tag-outline', 'tag-multiple', 'folder', 'folder-outline', 'folder-open', 'file', 'file-outline', 'file-document', 'image', 'image-outline', 'image-multiple', 'video', 'video-outline', 'video-stabilization', 'music', 'music-note', 'music-outline', 'play', 'play-outline', 'play-circle', 'pause', 'pause-outline', 'pause-circle', 'stop', 'stop-circle', 'refresh', 'refresh-circle', 'sync', 'sync-alert', 'wifi', 'wifi-off', 'wifi-strength-1', 'bluetooth', 'bluetooth-connect', 'battery', 'battery-outline', 'battery-charging', 'signal', 'signal-cellular-outline', 'notification', 'bell', 'bell-outline', 'bell-ring', 'message', 'message-outline', 'message-text', 'chat', 'chat-outline', 'comment', 'comment-outline', 'comment-text', 'like', 'thumb-up', 'thumb-up-outline', 'dislike', 'thumb-down', 'thumb-down-outline', 'share', 'share-outline', 'share-variant', 'link', 'link-variant', 'link-variant-outline', 'copy', 'content-copy', 'paste', 'content-paste', 'cut', 'content-cut', 'undo', 'undo-variant', 'redo', 'redo-variant', 'save', 'content-save', 'content-save-outline', 'print', 'printer', 'printer-outline', 'filter', 'filter-outline', 'filter-variant', 'sort', 'sort-alphabetical', 'sort-numeric', 'grid', 'view-grid', 'view-grid-outline', 'list', 'view-list', 'view-list-outline', 'zoom-in', 'magnify-plus', 'zoom-out', 'magnify-minus', 'fullscreen', 'fullscreen-exit', 'minimize', 'window-minimize', 'maximize', 'window-maximize', 'window-restore', 'chevron-up', 'chevron-down', 'chevron-left', 'chevron-right', 'chevron-up-circle', 'chevron-down-circle', 'chevron-left-circle', 'chevron-right-circle', 'dots-vertical', 'dots-horizontal', 'more-vertical', 'more-horizontal'],
            'fa6': ['star', 'star-half-stroke', 'heart', 'heart-crack', 'check', 'circle-check', 'arrow-right', 'arrow-left', 'arrow-up', 'arrow-down', 'arrow-right-long', 'arrow-left-long', 'house', 'house-chimney', 'user', 'user-large', 'gear', 'gears', 'magnifying-glass', 'bars', 'xmark', 'xmark-circle', 'pen', 'pen-to-square', 'trash', 'trash-can', 'plus', 'circle-plus', 'minus', 'circle-minus', 'download', 'file-arrow-down', 'upload', 'file-arrow-up', 'lock', 'unlock', 'eye', 'eye-slash', 'calendar', 'calendar-days', 'clock', 'clock-four', 'envelope', 'envelope-open', 'phone', 'phone-flip', 'map', 'map-location-dot', 'cart-shopping', 'shopping-cart', 'credit-card', 'shield', 'shield-halved', 'trophy', 'award', 'fire', 'fire-flame-curved', 'bolt', 'rocket', 'flag', 'bookmark', 'tag', 'folder', 'folder-open', 'file', 'file-lines', 'image', 'images', 'video', 'video-camera', 'music', 'play', 'pause', 'stop', 'rotate', 'rotate-right', 'sync', 'sync-alt', 'wifi', 'bluetooth', 'battery-full', 'battery-three-quarters', 'signal', 'signal-5', 'bell', 'bell-slash', 'message', 'comments', 'comment', 'thumbs-up', 'thumbs-down', 'share', 'share-nodes', 'link', 'copy', 'paste', 'scissors', 'undo', 'redo', 'floppy-disk', 'print', 'filter', 'sort', 'sort-up', 'sort-down', 'grid', 'grid-3', 'list', 'list-ul', 'magnifying-glass-plus', 'magnifying-glass-minus', 'maximize', 'minimize', 'window-restore', 'chevron-up', 'chevron-down', 'chevron-left', 'chevron-right', 'arrows-up-down', 'arrows-left-right', 'arrows-rotate', 'grip-vertical', 'grip-horizontal', 'ellipsis-vertical', 'ellipsis'],
            'bi': ['star', 'star-fill', 'star-half', 'heart', 'heart-fill', 'heart-pulse-fill', 'check', 'check-circle', 'check-circle-fill', 'check-all', 'arrow-right', 'arrow-left', 'arrow-up', 'arrow-down', 'arrow-right-circle', 'arrow-left-circle', 'house', 'house-door', 'house-fill', 'person', 'person-fill', 'person-circle', 'gear', 'gear-fill', 'search', 'list', 'x', 'x-circle', 'x-circle-fill', 'pencil', 'pencil-fill', 'pencil-square', 'trash', 'trash-fill', 'plus', 'plus-circle', 'plus-circle-fill', 'dash', 'dash-circle', 'download', 'download-cloud', 'upload', 'upload-cloud', 'lock', 'lock-fill', 'unlock', 'unlock-fill', 'eye', 'eye-fill', 'eye-slash', 'eye-slash-fill', 'calendar', 'calendar-event', 'calendar-fill', 'clock', 'clock-fill', 'clock-history', 'envelope', 'envelope-fill', 'envelope-open', 'telephone', 'telephone-fill', 'geo-alt', 'geo-alt-fill', 'cart', 'cart-fill', 'cart-plus', 'credit-card', 'credit-card-fill', 'shield', 'shield-fill', 'shield-check', 'trophy', 'trophy-fill', 'award', 'award-fill', 'fire', 'lightning', 'lightning-fill', 'rocket', 'rocket-fill', 'rocket-takeoff', 'flag', 'flag-fill', 'bookmark', 'bookmark-fill', 'tag', 'tag-fill', 'folder', 'folder-fill', 'folder-open', 'file', 'file-earmark', 'file-earmark-fill', 'image', 'image-fill', 'images', 'camera-video', 'camera-video-fill', 'music-player', 'music-player-fill', 'play', 'play-fill', 'play-circle', 'pause', 'pause-fill', 'pause-circle', 'stop', 'stop-fill', 'stop-circle', 'arrow-clockwise', 'arrow-repeat', 'arrow-counterclockwise', 'wifi', 'wifi-off', 'bluetooth', 'battery-full', 'battery-charging', 'signal', 'signal-4', 'bell', 'bell-fill', 'bell-slash', 'chat', 'chat-fill', 'chat-dots', 'chat-dots-fill', 'hand-thumbs-up', 'hand-thumbs-up-fill', 'hand-thumbs-down', 'hand-thumbs-down-fill', 'share', 'share-fill', 'link', 'link-45deg', 'files', 'clipboard', 'clipboard-fill', 'scissors', 'arrow-counterclockwise', 'arrow-clockwise', 'floppy', 'floppy-fill', 'printer', 'printer-fill', 'funnel', 'funnel-fill', 'sort-down', 'sort-up', 'sort-down-alt', 'grid', 'grid-3x3', 'grid-3x3-gap', 'list-ul', 'list-ol', 'zoom-in', 'zoom-out', 'fullscreen', 'fullscreen-exit', 'arrows-angle-contract', 'arrows-angle-expand', 'chevron-up', 'chevron-down', 'chevron-left', 'chevron-right', 'chevron-up-circle', 'chevron-down-circle', 'arrow-up', 'arrow-down', 'arrow-left', 'arrow-right', 'arrows-move', 'arrows-expand', 'three-dots-vertical', 'three-dots'],
            'heroicons': ['star', 'star-solid', 'heart', 'heart-solid', 'check', 'check-circle', 'arrow-right', 'arrow-left', 'arrow-up', 'arrow-down', 'arrow-long-right', 'arrow-long-left', 'home', 'home-solid', 'user', 'user-solid', 'cog-6-tooth', 'cog-6-tooth-solid', 'magnifying-glass', 'bars-3', 'x-mark', 'x-circle', 'pencil', 'pencil-square', 'trash', 'trash-solid', 'plus', 'plus-circle', 'minus', 'minus-circle', 'arrow-down-tray', 'arrow-up-tray', 'lock-closed', 'lock-open', 'eye', 'eye-slash', 'calendar', 'calendar-days', 'clock', 'envelope', 'envelope-open', 'phone', 'map-pin', 'map-pin-solid', 'shopping-cart', 'shopping-cart-solid', 'credit-card', 'credit-card-solid', 'shield-check', 'trophy', 'fire', 'bolt', 'rocket-launch', 'flag', 'bookmark', 'bookmark-solid', 'tag', 'folder', 'folder-solid', 'document', 'document-text', 'photo', 'video-camera', 'musical-note', 'play', 'play-circle', 'pause', 'pause-circle', 'stop', 'arrow-path', 'wifi', 'bluetooth', 'battery-100', 'signal', 'bell', 'bell-slash', 'chat-bubble-left', 'chat-bubble-left-right', 'hand-thumb-up', 'hand-thumb-down', 'share', 'link', 'clipboard-document', 'clipboard', 'scissors', 'arrow-uturn-left', 'arrow-uturn-right', 'document-arrow-down', 'printer', 'funnel', 'bars-arrow-up', 'squares-2x2', 'list-bullet', 'magnifying-glass-plus', 'magnifying-glass-minus', 'arrows-pointing-out', 'arrows-pointing-in', 'chevron-up', 'chevron-down', 'chevron-left', 'chevron-right', 'arrow-up', 'arrow-down', 'arrow-left', 'arrow-right', 'arrows-right-left', 'arrows-up-down', 'ellipsis-vertical', 'ellipsis-horizontal']
        };
        
        return allIcons[library] || [];
    }

    /**
     * Configurar scroll infinito
     */
    function setupInfiniteScroll() {
        // Usar MutationObserver para garantir que o listener seja anexado mesmo após o DOM ser modificado
        const observer = new MutationObserver(function(mutations) {
            const $results = $('#upgram-icon-selector-results');
            if ($results.length > 0 && $results.data('scroll-attached') !== true) {
                // Remover listeners antigos
                $results.off('scroll.upgram-infinite');
                
                // Anexar listener diretamente ao elemento (scroll não faz bubbling)
                $results.on('scroll.upgram-infinite', function() {
                    const element = this;
                    const scrollTop = element.scrollTop;
                    const scrollHeight = element.scrollHeight;
                    const clientHeight = element.clientHeight;
                    
                    // Verificar se está próximo do final (85% scrollado)
                    const scrollPercent = (scrollTop + clientHeight) / scrollHeight;
                    
                    if (scrollPercent >= 0.85 && hasMoreIcons && !isLoading) {
                        const query = $('#upgram-icon-selector-search').val().trim();
                        console.log('Scroll infinito: carregando mais ícones...', {
                            scrollTop: scrollTop,
                            scrollHeight: scrollHeight,
                            clientHeight: clientHeight,
                            scrollPercent: (scrollPercent * 100).toFixed(1) + '%',
                            hasMoreIcons: hasMoreIcons,
                            isLoading: isLoading
                        });
                        searchIcons(query || '', false);
                    }
                });
                
                // Marcar como anexado
                $results.data('scroll-attached', true);
            }
        });
        
        // Observar mudanças no body para garantir que o elemento seja encontrado
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
        
        // Tentar anexar imediatamente se o elemento já existir
        setTimeout(function() {
            const $results = $('#upgram-icon-selector-results');
            if ($results.length > 0) {
                $results.off('scroll.upgram-infinite');
                $results.on('scroll.upgram-infinite', function() {
                    const element = this;
                    const scrollTop = element.scrollTop;
                    const scrollHeight = element.scrollHeight;
                    const clientHeight = element.clientHeight;
                    const scrollPercent = (scrollTop + clientHeight) / scrollHeight;
                    
                    if (scrollPercent >= 0.85 && hasMoreIcons && !isLoading) {
                        const query = $('#upgram-icon-selector-search').val().trim();
                        console.log('Scroll infinito: carregando mais ícones...');
                        searchIcons(query || '', false);
                    }
                });
                $results.data('scroll-attached', true);
            }
        }, 100);
    }

    /**
     * Buscar ícones com paginação
     */
    function searchIcons(query, reset = false) {
        const library = $('#upgram-icon-selector-library').val();
        
        if (reset) {
            currentPage = 0;
            currentQuery = query;
            currentLibrary = library;
            hasMoreIcons = true;
            $('#upgram-icon-selector-results').html('');
        }

        if (isLoading || !hasMoreIcons) {
            return;
        }

        isLoading = true;
        const cacheKey = `${library}_${query}_${currentPage}`;

        // Verificar cache
        if (searchCache[cacheKey]) {
            appendIcons(searchCache[cacheKey], reset);
            isLoading = false;
            return;
        }

        // Obter todos os ícones da biblioteca
        const allIcons = getAllPopularIcons(library);
        
        // Filtrar ícones por query
        let filteredIcons = allIcons.filter(icon => 
            icon.toLowerCase().includes(query.toLowerCase())
        );

        // Paginação: 60 ícones por página
        const iconsPerPage = 60;
        const startIndex = currentPage * iconsPerPage;
        const endIndex = startIndex + iconsPerPage;
        const icons = filteredIcons.slice(startIndex, endIndex);

        if (icons.length === 0) {
            if (currentPage === 0) {
                $('#upgram-icon-selector-results').html('<div style="text-align: center; padding: 40px; color: #999;">Nenhum ícone encontrado. Tente outra palavra-chave.</div>');
            }
            hasMoreIcons = false;
            isLoading = false;
            return;
        }

        // Verificar se há mais ícones
        hasMoreIcons = endIndex < filteredIcons.length;

        // Cachear resultado
        searchCache[cacheKey] = icons;

        // Exibir ícones
        appendIcons(icons, reset);
        isLoading = false;
        currentPage++;
    }

    /**
     * Adicionar ícones à lista (com scroll infinito)
     */
    function appendIcons(icons, reset) {
        const library = $('#upgram-icon-selector-library').val();
        const $results = $('#upgram-icon-selector-results');
        
        // Se reset, limpar container
        if (reset) {
            $results.html('<div id="upgram-icons-container" style="display: flex; flex-wrap: wrap; gap: 10px; justify-content: flex-start;"></div>');
        }

        const $container = $('#upgram-icons-container');
        if ($container.length === 0) {
            $results.html('<div id="upgram-icons-container" style="display: flex; flex-wrap: wrap; gap: 10px; justify-content: flex-start;"></div>');
        }

        console.log('appendIcons chamado com', icons.length, 'ícones, reset:', reset);
        
        // Verificar se Iconify está disponível
        if (typeof Iconify === 'undefined' || typeof Iconify.getIcon !== 'function') {
            console.log('Iconify não disponível, tentando carregar...');
            // Tentar carregar Iconify
            loadIconify().then(function(IconifyAPI) {
                console.log('Iconify carregado com sucesso, renderizando ícones...');
                renderIcons(icons, library);
            }).catch(function(error) {
                console.error('Erro ao carregar Iconify:', error);
                renderIconsWithFallback(icons, library);
            });
        } else {
            // Iconify já está disponível
            console.log('Iconify já disponível, renderizando ícones...');
            renderIcons(icons, library);
        }

        function renderIcons(icons, library) {
            console.log('renderIcons: renderizando', icons.length, 'ícones da biblioteca', library);
            let loadedCount = 0;
            const totalIcons = icons.length;

            if (totalIcons === 0) {
                console.warn('Nenhum ícone para renderizar');
                return;
            }

            icons.forEach(function(icon, index) {
                const iconName = library + ':' + icon;
                const itemId = 'upgram-icon-item-' + Date.now() + '-' + index;
                
                // Criar elemento do ícone
                const $item = $('<div>', {
                    id: itemId,
                    class: 'upgram-icon-item',
                    'data-icon': iconName,
                    css: {
                        'display': 'inline-block',
                        'width': '100px',
                        'padding': '15px 10px',
                        'margin': '5px',
                        'text-align': 'center',
                        'border': '2px solid #ddd',
                        'border-radius': '6px',
                        'cursor': 'pointer',
                        'background': 'white',
                        'transition': 'all 0.2s',
                        'position': 'relative',
                        'min-height': '100px'
                    }
                });

                // Container do preview
                const $preview = $('<div>', {
                    class: 'upgram-icon-preview',
                    css: {
                        'width': '32px',
                        'height': '32px',
                        'margin': '0 auto 8px',
                        'display': 'flex',
                        'align-items': 'center',
                        'justify-content': 'center'
                    }
                });

                // Buscar SVG do ícone via API HTTP (mais confiável que Iconify.getIcon)
                // Parsear nome do ícone (ex: "mdi:star" -> prefix: "mdi", name: "star")
                const parts = iconName.split(':');
                if (parts.length === 2) {
                    const prefix = parts[0];
                    const name = parts[1];
                    const apiUrl = 'https://api.iconify.design/' + prefix + '/' + name + '.svg';
                    
                    // Buscar SVG via fetch
                    fetch(apiUrl)
                        .then(function(response) {
                            if (!response.ok) {
                                throw new Error('HTTP error! status: ' + response.status);
                            }
                            return response.text();
                        })
                        .then(function(svgText) {
                            if (!svgText || svgText.trim() === '') {
                                throw new Error('SVG vazio');
                            }
                            
                            // Verificar se é um SVG válido
                            if (svgText.indexOf('<svg') === -1) {
                                throw new Error('Resposta não é um SVG válido');
                            }
                            
                            // Criar um elemento temporário para processar o SVG
                            const tempDiv = document.createElement('div');
                            tempDiv.innerHTML = svgText.trim();
                            const svgElement = tempDiv.querySelector('svg');
                            
                            if (svgElement) {
                                // Configurar atributos do SVG
                                svgElement.setAttribute('width', '32');
                                svgElement.setAttribute('height', '32');
                                svgElement.setAttribute('fill', 'currentColor');
                                svgElement.style.color = '#333';
                                svgElement.style.display = 'block';
                                
                                // Limpar preview e adicionar SVG
                                $preview.empty();
                                $preview.append(svgElement);
                                
                                // Adicionar nome do ícone
                                const $name = $('<div>', {
                                    style: 'font-size: 10px; margin-top: 5px; color: #666; word-break: break-word;'
                                }).text(icon);
                                
                                $item.append($preview).append($name);
                                $container.append($item);
                                
                                loadedCount++;
                                
                                // Configurar eventos quando todos os ícones forem carregados
                                if (loadedCount === totalIcons) {
                                    setupIconEvents();
                                }
                            } else {
                                throw new Error('SVG element não encontrado');
                            }
                        })
                        .catch(function(error) {
                            console.error('Erro ao buscar ícone via API:', iconName, error);
                            // Fallback: usar tag iconify-icon
                            $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                            
                            // Adicionar nome do ícone
                            const $name = $('<div>', {
                                style: 'font-size: 10px; margin-top: 5px; color: #666; word-break: break-word;'
                            }).text(icon);
                            
                            $item.append($preview).append($name);
                            $container.append($item);
                            
                            loadedCount++;
                            
                            if (loadedCount === totalIcons) {
                                setupIconEvents();
                                // Tentar escanear ícones do Iconify se disponível
                                if (typeof Iconify !== 'undefined' && typeof Iconify.scan === 'function') {
                                    Iconify.scan($container[0]);
                                }
                            }
                        });
                } else {
                    // Formato inválido, usar fallback
                    console.warn('Formato de nome de ícone inválido:', iconName);
                    $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                    
                    const $name = $('<div>', {
                        style: 'font-size: 10px; margin-top: 5px; color: #666; word-break: break-word;'
                    }).text(icon);
                    
                    $item.append($preview).append($name);
                    $container.append($item);
                    
                    loadedCount++;
                    
                    if (loadedCount === totalIcons) {
                        setupIconEvents();
                        if (typeof Iconify !== 'undefined' && typeof Iconify.scan === 'function') {
                            Iconify.scan($container[0]);
                        }
                    }
                }
            });
        }

        function renderIconsWithFallback(icons, library) {
            // Renderizar com tag iconify-icon como fallback
            const $container = $('#upgram-icons-container');
            icons.forEach(function(icon, index) {
                const iconName = library + ':' + icon;
                const $item = $('<div>', {
                    class: 'upgram-icon-item',
                    'data-icon': iconName,
                    css: {
                        'display': 'inline-block',
                        'width': '100px',
                        'padding': '15px 10px',
                        'margin': '5px',
                        'text-align': 'center',
                        'border': '2px solid #ddd',
                        'border-radius': '6px',
                        'cursor': 'pointer',
                        'background': 'white'
                    }
                }).html('<iconify-icon icon="' + iconName + '" style="font-size: 32px;"></iconify-icon><div style="font-size: 10px; margin-top: 5px;">' + icon + '</div>');
                $container.append($item);
            });
            setupIconEvents();
        }
    }

    /**
     * Configurar eventos dos ícones
     */
    function setupIconEvents() {
        // Remover listeners antigos e adicionar novos usando event delegation
        $(document).off('click', '.upgram-icon-item');
        $(document).on('click', '.upgram-icon-item', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            
            const $item = $(this);
            const iconName = $item.attr('data-icon');
            
            if (!iconName) {
                console.error('Nome do ícone não encontrado no elemento:', $item);
                return false;
            }
            
            console.log('Ícone clicado:', iconName);
            
            // Feedback visual
            $item.css({
                'background': '#e3f2fd',
                'border-color': '#7C51F3'
            });
            
            // Selecionar ícone imediatamente
            setTimeout(function() {
                selectIcon(iconName);
            }, 50);
            
            return false;
        });

        // Hover effect - usar event delegation também
        $(document).off('mouseenter mouseleave', '.upgram-icon-item');
        $(document).on('mouseenter', '.upgram-icon-item', function() {
            $(this).css({
                'background': '#f0f0f0',
                'border-color': '#7C51F3',
                'transform': 'scale(1.05)',
                'box-shadow': '0 2px 8px rgba(124, 81, 243, 0.3)'
            });
        }).on('mouseleave', '.upgram-icon-item', function() {
            $(this).css({
                'background': 'white',
                'border-color': '#ddd',
                'transform': 'scale(1)',
                'box-shadow': 'none'
            });
        });
    }

    /**
     * Selecionar ícone
     */
    function selectIcon(iconName) {
        console.log('selectIcon chamado com:', iconName);
        console.log('currentTargetField:', currentTargetField);
        
        if (!currentTargetField) {
            console.error('Campo de destino não definido');
            showNotification('Erro: Campo de destino não encontrado');
            return;
        }

        if (!iconName) {
            console.error('Nome do ícone não fornecido');
            showNotification('Erro: Nome do ícone não fornecido');
            return;
        }

        console.log('Selecionando ícone:', iconName);

        // Mostrar loading
        showNotification('Convertendo ícone para SVG...');

        // Verificar se Iconify está disponível
        if (typeof Iconify === 'undefined') {
            console.warn('Iconify não está definido, usando fallback');
            useIconifyTag(iconName);
            return;
        }

        if (typeof Iconify.getIcon !== 'function') {
            console.warn('Iconify.getIcon não é uma função, usando fallback');
            useIconifyTag(iconName);
            return;
        }

        // Usar API HTTP diretamente (mais confiável que Iconify.getIcon)
        fetchIconViaAPI(iconName);
    }

    /**
     * Buscar ícone via API HTTP do Iconify
     */
    function fetchIconViaAPI(iconName) {
        console.log('Buscando ícone via API HTTP:', iconName);
        
        // Parsear nome do ícone (ex: "mdi:star" -> prefix: "mdi", name: "star")
        const parts = iconName.split(':');
        if (parts.length !== 2) {
            console.error('Formato de nome de ícone inválido:', iconName);
            useIconifyTag(iconName);
            return;
        }
        
        const prefix = parts[0];
        const name = parts[1];
        
        // API do Iconify: https://api.iconify.design/{prefix}/{name}.svg
        const apiUrl = 'https://api.iconify.design/' + prefix + '/' + name + '.svg';
        
        fetch(apiUrl)
            .then(function(response) {
                if (!response.ok) {
                    throw new Error('HTTP error! status: ' + response.status);
                }
                return response.text();
            })
            .then(function(svgText) {
                if (!svgText || svgText.trim() === '') {
                    throw new Error('SVG vazio retornado da API');
                }
                
                // Verificar se é um SVG válido
                if (svgText.indexOf('<svg') === -1) {
                    throw new Error('Resposta não é um SVG válido');
                }
                
                // Inserir SVG no campo
                if (currentTargetField) {
                    currentTargetField.value = svgText.trim();
                    $(currentTargetField).trigger('change').trigger('input');
                    
                    // Fechar modal
                    closeIconSelector();
                    
                    // Mostrar mensagem de sucesso
                    showNotification('✓ Ícone selecionado com sucesso!');
                }
            })
            .catch(function(error) {
                console.error('Erro ao buscar ícone via API:', error);
                // Fallback para tag iconify-icon
                useIconifyTag(iconName);
            });
    }

    /**
     * Usar tag iconify-icon como fallback
     */
    function useIconifyTag(iconName) {
        console.log('Usando tag iconify-icon como fallback para:', iconName);
        const iconHtml = '<iconify-icon icon="' + iconName + '" style="font-size: 32px;"></iconify-icon>';
        if (currentTargetField) {
            currentTargetField.value = iconHtml;
            $(currentTargetField).trigger('change').trigger('input');
            closeIconSelector();
            showNotification('Ícone selecionado! (Usando tag iconify-icon - certifique-se de que o Iconify está carregado na página do produto)');
        }
    }

    /**
     * Fechar modal
     */
    function closeIconSelector() {
        $('#upgram-icon-selector-modal').fadeOut(200);
        currentTargetField = null;
    }

    /**
     * Mostrar notificação
     */
    function showNotification(message) {
        // Remover notificações antigas
        $('.upgram-icon-notification').remove();
        
        // Criar notificação
        const $notification = $('<div>', {
            class: 'upgram-icon-notification',
            text: message,
            css: {
                'position': 'fixed',
                'top': '20px',
                'right': '20px',
                'background': '#333',
                'color': 'white',
                'padding': '12px 20px',
                'border-radius': '4px',
                'z-index': '100001',
                'box-shadow': '0 2px 10px rgba(0,0,0,0.3)'
            }
        });
        
        $('body').append($notification);
        
        // Remover após 3 segundos
        setTimeout(function() {
            $notification.fadeOut(200, function() {
                $(this).remove();
            });
        }, 3000);
    }

    // Inicializar quando o DOM estiver pronto
    $(document).ready(function() {
        initIconSelector();
        
        // Re-inicializar quando campos são adicionados dinamicamente
        $(document).on('DOMNodeInserted', function(e) {
            if ($(e.target).find('.upgram-icon-field, .upgram-eeat-icon-field, .upgram-years-icon-field, .upgram-orders-icon-field').length > 0) {
                initIconSelector();
            }
        });
    });

})(jQuery);
                    try {
                        const iconPromise = Iconify.getIcon(iconName);
                        
                        if (iconPromise && typeof iconPromise.then === 'function') {
                            iconPromise.then(function(iconData) {
                                if (iconData && iconData.body) {
                                    try {
                                        // Criar SVG inline
                                        const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                                        svg.setAttribute('width', '32');
                                        svg.setAttribute('height', '32');
                                        svg.setAttribute('viewBox', '0 0 ' + (iconData.width || 24) + ' ' + (iconData.height || 24));
                                        svg.setAttribute('fill', 'currentColor');
                                        svg.style.color = '#333';
                                        svg.style.display = 'block';

                                        // Processar body do ícone
                                        const parser = new DOMParser();
                                        const doc = parser.parseFromString('<svg>' + iconData.body + '</svg>', 'image/svg+xml');
                                        const svgElements = doc.querySelector('svg');
                                        
                                        if (svgElements && svgElements.childNodes.length > 0) {
                                            Array.from(svgElements.childNodes).forEach(function(node) {
                                                if (node.nodeType === 1) {
                                                    const cloned = node.cloneNode(true);
                                                    svg.appendChild(cloned);
                                                }
                                            });
                                            $preview.html(svg);
                                        } else {
                                            // Fallback: usar tag iconify-icon
                                            $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                                        }
                                    } catch (e) {
                                        console.error('Erro ao criar SVG preview:', e);
                                        $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                                    }
                                } else {
                                    // Fallback: usar tag iconify-icon
                                    $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                                }
                            }).catch(function(error) {
                                console.error('Erro ao obter ícone para preview:', error);
                                // Fallback em caso de erro
                                $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                            });
                        } else {
                            // Não retornou Promise válida, usar fallback
                            $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                        }
                    } catch (e) {
                        console.error('Erro ao chamar Iconify.getIcon:', e);
                        $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                    }
                } else {
                    // Iconify não disponível ainda, usar tag
                    $preview.html('<iconify-icon icon="' + iconName + '" style="font-size: 32px; color: #333;"></iconify-icon>');
                }

                // Nome do ícone
                const $name = $('<div>', {
                    text: icon,
                    css: {
                        'font-size': '10px',
                        'color': '#666',
                        'word-break': 'break-word',
                        'line-height': '1.3'
                    }
                });

                $item.append($preview).append($name);
                $container.append($item);

                loadedCount++;
                if (loadedCount === totalIcons) {
                    // Todos os ícones carregados, configurar eventos
                    setupIconEvents();
                    
                    // Forçar renderização do Iconify se necessário
                    if (typeof Iconify !== 'undefined' && Iconify.scan) {
                        Iconify.scan($container[0]);
                    }
                }
            });
        }

        function renderIconsWithFallback(icons, library) {
            // Renderizar com tag iconify-icon como fallback
            const $container = $('#upgram-icons-container');
            icons.forEach(function(icon, index) {
                const iconName = library + ':' + icon;
                const $item = $('<div>', {
                    class: 'upgram-icon-item',
                    'data-icon': iconName,
                    css: {
                        'display': 'inline-block',
                        'width': '100px',
                        'padding': '15px 10px',
                        'margin': '5px',
                        'text-align': 'center',
                        'border': '2px solid #ddd',
                        'border-radius': '6px',
                        'cursor': 'pointer',
                        'background': 'white'
                    }
                }).html('<iconify-icon icon="' + iconName + '" style="font-size: 32px;"></iconify-icon><div style="font-size: 10px; margin-top: 5px;">' + icon + '</div>');
                $container.append($item);
            });
            setupIconEvents();
        }
    }

    /**
     * Configurar eventos dos ícones
     */
    function setupIconEvents() {
        // Remover listeners antigos e adicionar novos usando event delegation
        $(document).off('click', '.upgram-icon-item');
        $(document).on('click', '.upgram-icon-item', function(e) {
            e.preventDefault();
            e.stopPropagation();
            e.stopImmediatePropagation();
            
            const $item = $(this);
            const iconName = $item.attr('data-icon');
            
            if (!iconName) {
                console.error('Nome do ícone não encontrado no elemento:', $item);
                return false;
            }
            
            console.log('Ícone clicado:', iconName);
            
            // Feedback visual
            $item.css({
                'background': '#e3f2fd',
                'border-color': '#7C51F3'
            });
            
            // Selecionar ícone imediatamente
            selectIcon(iconName);
            
            return false;
        });

        // Hover effect - usar event delegation também
        $(document).off('mouseenter mouseleave', '.upgram-icon-item');
        $(document).on('mouseenter', '.upgram-icon-item', function() {
            $(this).css({
                'background': '#f0f0f0',
                'border-color': '#7C51F3',
                'transform': 'scale(1.05)',
                'box-shadow': '0 2px 8px rgba(124, 81, 243, 0.3)'
            });
        }).on('mouseleave', '.upgram-icon-item', function() {
            $(this).css({
                'background': 'white',
                'border-color': '#ddd',
                'transform': 'scale(1)',
                'box-shadow': 'none'
            });
        });
    }

    /**
     * Selecionar ícone
     */
    function selectIcon(iconName) {
        console.log('selectIcon chamado com:', iconName);
        console.log('currentTargetField:', currentTargetField);
        
        if (!currentTargetField) {
            console.error('Campo de destino não definido');
            showNotification('Erro: Campo de destino não encontrado');
            return;
        }

        if (!iconName) {
            console.error('Nome do ícone não fornecido');
            showNotification('Erro: Nome do ícone não fornecido');
            return;
        }

        console.log('Selecionando ícone:', iconName);

        // Mostrar loading
        showNotification('Convertendo ícone para SVG...');

        // Verificar se Iconify está disponível
        if (typeof Iconify === 'undefined') {
            console.warn('Iconify não está definido, usando fallback');
            useIconifyTag(iconName);
            return;
        }

        if (typeof Iconify.getIcon !== 'function') {
            console.warn('Iconify.getIcon não é uma função, usando fallback');
            useIconifyTag(iconName);
            return;
        }

        // Usar API HTTP diretamente (mais confiável que Iconify.getIcon)
        fetchIconViaAPI(iconName);
    }

    /**
     * Buscar ícone via API HTTP do Iconify
     */
    function fetchIconViaAPI(iconName) {
        console.log('Buscando ícone via API HTTP:', iconName);
        
        // Parsear nome do ícone (ex: "mdi:star" -> prefix: "mdi", name: "star")
        const parts = iconName.split(':');
        if (parts.length !== 2) {
            console.error('Formato de nome de ícone inválido:', iconName);
            useIconifyTag(iconName);
            return;
        }
        
        const prefix = parts[0];
        const name = parts[1];
        
        // API do Iconify: https://api.iconify.design/{prefix}/{name}.svg
        const apiUrl = 'https://api.iconify.design/' + prefix + '/' + name + '.svg';
        
        fetch(apiUrl)
            .then(function(response) {
                if (!response.ok) {
                    throw new Error('HTTP error! status: ' + response.status);
                }
                return response.text();
            })
            .then(function(svgText) {
                if (!svgText || svgText.trim() === '') {
                    throw new Error('SVG vazio retornado da API');
                }
                
                // Verificar se é um SVG válido
                if (svgText.indexOf('<svg') === -1) {
                    throw new Error('Resposta não é um SVG válido');
                }
                
                // Inserir SVG no campo
                if (currentTargetField) {
                    currentTargetField.value = svgText.trim();
                    $(currentTargetField).trigger('change').trigger('input');
                    
                    // Fechar modal
                    closeIconSelector();
                    
                    // Mostrar mensagem de sucesso
                    showNotification('✓ Ícone selecionado com sucesso!');
                }
            })
            .catch(function(error) {
                console.error('Erro ao buscar ícone via API:', error);
                // Fallback para tag iconify-icon
                useIconifyTag(iconName);
            });
    }

    /**
     * Usar tag iconify-icon como fallback
     */
    function useIconifyTag(iconName) {
        console.log('Usando tag iconify-icon como fallback para:', iconName);
        const iconHtml = '<iconify-icon icon="' + iconName + '" style="font-size: 32px;"></iconify-icon>';
        if (currentTargetField) {
            currentTargetField.value = iconHtml;
            $(currentTargetField).trigger('change').trigger('input');
            closeIconSelector();
            showNotification('Ícone selecionado! (Usando tag iconify-icon - certifique-se de que o Iconify está carregado na página do produto)');
        }
    }

    /**
     * Fechar modal
     */
    function closeIconSelector() {
        $('#upgram-icon-selector-modal').fadeOut(200);
        currentTargetField = null;
    }

    /**
     * Mostrar notificação
     */
    function showNotification(message) {
        const $notification = $('<div style="position: fixed; top: 20px; right: 20px; background: #4CAF50; color: white; padding: 15px 20px; border-radius: 4px; box-shadow: 0 4px 12px rgba(0,0,0,0.2); z-index: 100001; font-size: 14px;">' + message + '</div>');
        $('body').append($notification);
        setTimeout(() => {
            $notification.fadeOut(300, function() {
                $(this).remove();
            });
        }, 3000);
    }

    // Inicializar quando o DOM estiver pronto
    $(document).ready(function() {
        initIconSelector();
        
        // Re-inicializar quando campos são adicionados dinamicamente
        $(document).on('DOMNodeInserted', function(e) {
            if ($(e.target).find('.upgram-icon-field, .upgram-eeat-icon-field, .upgram-years-icon-field, .upgram-orders-icon-field').length > 0) {
                initIconSelector();
            }
        });
    });

})(jQuery);

